<?php
/*
 * $Id: standard.php,v 1.1 2007/09/23 20:04:34 jm Exp $
 * 
 * Created on 15 mai 07 by AEL Sarl for
 * 
 * osCommerce, Open Source E-Commerce Solutions
 * http://www.oscommerce.com
 *
 * Copyright (c) 2003 osCommerce
 *
 * Released under the GNU General Public License
 * 
 */
 
require(DIR_WS_CLASSES . 'shipping.php');

class manager {
	/**
	 * A class handling special values
	 */
	
	var $reference, $account_name, $account_number, $next_handler;
	var $value, $total_value;
	
	function manager($reference, $account_name, $account_number, $next_handler=null) {
		$this->reference = $reference;
		$this->account_number = $account_number;
		$this->account_name = $account_name;
		$this->next_handler = $next_handler;
		$this->value = 0;
	}
	
	function set_next_handler($next) {
		if (is_a($next, 'manager'))
			$this->next_handler = $next;
	}
	
	function handle_value($ref, $val) {
		if ($ref == $this->reference)
			$this->value += $val;
		else if($this->next_handler != null && is_a($this->next_handler, 'manager'))
			$this->next_handler->handle_value($ref, $val);
	}
	
	function has_handler($ref) {
		/**
		 * Check if we have a handler of $reference in the chain
		 * @param			ref the reference value we want to check the handling of
		 * @return		a boolean flag
		 */
		$handler = $this;
		while ($handler != null) {
			if ($handler->reference == $ref)
				return true;
			$handler = $handler->next_handler;
		}
		return false;
	}
	
	function add_handler($new) {
		/**
		 * Add a handler in the chain
		 * @param new	handler to add
		 */
		$handler = &$this;
		while ($handler->next_handler != null)
			$handler = &$handler->next_handler;
		$handler->set_next_handler($new);
	}
	
	/**
	 * Add all stored values in the handlers chain 
	 * and store the result in the total_value attribute if requested
	 * @param store a flag indicating weather the result must be stored (if true) or not (if false, the default)
	 * @param round a flag indicating weather each value in the chain must be rounded or not (if false, the default)
	 */
	function sum($store=false, $round=false) {
		$handler = $this;
		$value = 0;
		while ($handler != null) {
			if (round)
				$value += round($handler->value, 2);
			else
				$value += $handler->value;
			$handler = $handler->next_handler;
		}
		if ($store != 0)
			$this->total_value = $value;
			
		return $value;
	}
	
	/**
	 * Reset values of the total_value and value attributes to 0,
	 * in all handlers of the chain
	 */
	function clear() {
		$handler = &$this;
		while ($handler != null) {
			$handler->total_value = 0;
			$handler->value = 0;
			$handler = & $handler->next_handler;
		}
	}
	
} 
 
/**
 * A class handling products taxes
 */
class taxManager extends manager{
	
	var $shipping_modules;
	
	function taxManager($tax_rate=19.6, $account_name="TVA 19.6%", $account_number="44571300", $next_handler=null) {
		parent::manager($tax_rate, $account_name, $account_number, $next_handler);
		$this->shipping_modules = array();
		$shippings = new shipping();
		foreach ($shippings->modules as $module) {
			$class = substr($module, 0, strrpos($module, '.'));
			$this->shipping_modules[] = new $class();
		}
	}
	
	/**
		 * handle shipping and returns some shipping informations
		 * @param $id the order id
		 * @return an array with the total shipping (included tax) cost and the shipping tax as keys
		 */
	function handle_shipping($id) {
		
		// get the total amount of taxes for the order we've the id
		$shipping_query = tep_db_query("select value, title from " . TABLE_ORDERS_TOTAL . 
							        						 " where orders_id=" . $id .
															     " and class = 'ot_shipping'");
		$shipping = tep_db_fetch_array($shipping_query);
		
		reset($this->shipping_modules);
		$shipping_tax_value = 0;
		$shipping_tax_rate = 0;
		foreach ($this->shipping_modules as $module) {
			if (strpos($shipping['title'], $module->title, 0) !== false) {
				$shipping_tax_rate = tep_get_tax_rate($module->tax_class);
				$shipping_tax_value = $shipping['value'] - ($shipping['value'] / (($shipping_tax_rate < 10) ? "1.0" . str_replace('.', '', $shipping_tax_rate) : "1." . str_replace('.', '', $shipping_tax_rate)));
				/*$fp = fopen('share.log', 'a');
				fwrite($fp, 'Trouve le port........' . "\n");
				fwrite($fp, 'total shipping : ' . $shipping['value'] . "\n");
				fwrite($fp, 'shipping tax rate : ' . $shipping_tax_rate . "\n");
				fwrite($fp, 'shipping tax value : ' . $shipping_tax_value . "\n");
				fclose($fp);*/
				break;
			}
		}
    
    return array('total_shipping' => $shipping['value'],
    						 'shipping_tax_rate' => $shipping_tax_rate,
                 'shipping_tax_value' => $shipping_tax_value);
	}
}
 
/**
 * standard module for invoices export
 * 
 * fabrique un fichier d'export dont le format est :
 * - rien pour le numéro de ligne
 * - date de facture (raw format)
 * - libellé du journal (option de configuration)
 * - numéro du compte comptable
 * - rien pour le libellé automatique
 * - nom du compte comptable entre guillemets
 * - numéro de facture entre guillemets ()
 * - montant HT
 * - sens de l'opération (D pour débit)
 * - rien pour la date d'échéance
 * - type de tva (D pour débit)
 * - devise de l'opération
 */
class standard {
	
	var $content_type, $key, $description, $shipping, $separator, $tax_manager, $sale_manager;
	var $accounts, $customer_account, $shipping_account; // arrays with account_name and account_number keys
	
	function standard($separator=',') {
		$this->content_type = "application/csv-tab-delimited-table";
		$this->key = 'standard';
		$this->description = 'Module standard';
		$this->separator = $separator;
		$this->tax_manager = null;
		$this->sale_manager = null;
		$this->accounts = array();
		
		// build the tax and sale managers from the database tables
		// create the customer and shipping accounts
		$accounts_query = tep_db_query("select tax_rate, account_name, ac.account_number, account_type, account_key" .
		                               " from " . TABLE_ACCOUNTS_CHART . " ac LEFT JOIN " . TABLE_TAX_RATES . " tr" .
		                               " ON tr.account_number = ac.account_number"  .
                                   " order by tr.tax_rate");
		while ($accounts = tep_db_fetch_array($accounts_query)) {
			$accounts['account_name'] = utf8_encode($accounts['account_name']);
			$this->accounts[$accounts['account_number']] = array('name' => $accounts['account_name'],
			                                                     'type' => $accounts['account_type']);
			if ($accounts['account_key'] == 'DEFAULT_CUS_ACCOUNT')
				$this->customer_account = array('number' => $accounts['account_number'],
				                                'name' => $accounts['account_name']);
			if ($accounts['account_key'] == 'DEFAULT_SHI_ACCOUNT')
				$this->shipping_account = array('number' => $accounts['account_number'],
				                                'name' => $accounts['account_name']);
			
			if ($accounts['account_type'] == 'TAX')	{	
				if ($this->tax_manager == null)
					$this->tax_manager = new taxManager($accounts['tax_rate'],
				                                      $accounts['account_name'],
				                                   	  $accounts['account_number']);
				else
					$this->tax_manager->add_handler(new taxManager($accounts['tax_rate'],
				                                                 $accounts['account_name'],
				                                    	           $accounts['account_number']));
			}
			else 	{ // means sale and shipping accounts	
				if ($this->sale_manager == null)
					$this->sale_manager = new manager($accounts['account_number'],
				                                    $accounts['account_name'],
				                                    $accounts['account_number']);
				else
					$this->sale_manager->add_handler(new manager($accounts['account_number'],
				                                               $accounts['account_name'],
				                                    	         $accounts['account_number']));
			}
		}
	}
	
	function _merge($merged) {
		$this->export = array_merge($this->export, $merged);
	}
	
	function _write_tax_account($row) {
		$lines = array();
		$handler = $this->tax_manager;
		$row['operation_mode'] = '';
		$row['tax_mode'] = 'E';
		while ($handler != null) {
			if ($handler->value != 0) {
				$row['account_name'] = $handler->account_name;
				$row['account_number'] = $handler->account_number;
				$row['products_excluded_tax'] = $handler->value;
				$lines[] = $this->_write_line($row);
			}
			$handler = $handler->next_handler;
		}
		
		return $lines;
	}
	
	function _write_shipping_account($row) {
		$lines = array();
		// write the shipping operation
		$row['operation_mode'] = 'C';
		$row['tax_mode'] = '';
		$row['account_name'] = $this->shipping_account['name'];
		$row['account_number'] = $this->shipping_account['number'];
		$lines[] = $this->_write_line($row);
		
		
		return $lines;
	}
	
	function _write_product_account($row) {
		$lines = array();
		$handler = $this->sale_manager;
		// write the sale
		$row['operation_mode'] = 'C';
		$row['tax_mode'] = '';
		while ($handler != null) {
			if ($handler->value != 0) {
				$row['account_name'] = $handler->account_name;
				$row['account_number'] = $handler->account_number;
				$row['products_excluded_tax'] = $handler->value;
				$lines[] = $this->_write_line($row);
			}
			$handler = $handler->next_handler;
		}
		
		return $lines;
	}
	
	function _write_customer_account($row) {
		$row['operation_mode'] = 'D';
		$row['tax_mode'] = '';
		$row['account_name'] = $this->customer_account['name'];
		$row['account_number'] = $this->customer_account['number'];
		$row['products_excluded_tax'] = $row['total_included_tax'];
		$lines[] = $this->_write_line($row);
		
		return $lines;
	}
	
	function _write_line($row) {
		return implode($this->separator,
									 array('',
									 			 tep_date_raw(tep_date_short($row['invoice_date']), $reverse=true),
									 			 SALES_JOURNAL,
									 			 $row['account_number'],
									 			 '',
									 			 '"' . $row['account_name'] . '"',
									 			 '"' . $row['invoice_number'] . '"',
									 			 sprintf("%.2f", tep_round($row['products_excluded_tax'], 2)),
									 			 $row['operation_mode'],
									 			 '',
									 			 $row['tax_mode'],
									       $row['currency']));
	}
	
	/**
	 * Manage an order's shipping operation:
	 * Fetch the shipping included tax part, calculate the shipping tax and adds it to the products taxes
	 * @param id the order id we want to manage the shipping part from
	 * @return the shipping excluded tax part
	 */
	function _manage_shipping($id) {
		
		 $shipping = $this->tax_manager->handle_shipping($id);
		 if ($shipping['shipping_tax_rate'] != 0) // shipping op has a tax rate, try to handle it
		 	 $this->tax_manager->handle_value($shipping['shipping_tax_rate'], $shipping['shipping_tax_value']);
		 $this->sale_manager->handle_value($this->shipping_account['number'],
		                                   $shipping['total_shipping'] - $shipping['shipping_tax_value']);
	}
	
	function _calculate_product_constants($price, $tax_rate, $quantity) {
		
		$included_tax = (round($price, 2) + round($price * $tax_rate / 100, 2)) * $quantity;
		$tax = round($included_tax - ($included_tax / (($tax_rate < 10) ? "1.0" . str_replace('.', '', $tax_rate) : "1." . str_replace('.', '', $tax_rate))), 2);
		$excluded_tax = $included_tax - $tax;
		return array('included_tax' => $included_tax,
		             'tax' => $tax,
		             'excluded_tax' => $excluded_tax);
	}
	
	/**
	 * Handle a row, managing it's values
	 * @param row the query row to handle
	 */
	function _handle_current_row(&$row) {
		$row['invoice_date'] = tep_date_raw(tep_date_short($row['invoice_date']), $reverse=true);
		// store the amount (excluded tax) part
		$constants = $this->_calculate_product_constants($row['final_price'],
		                                                 $row['products_tax'],
		                                                 $row['products_quantity']);
		$this->sale_manager->handle_value($row['account_number'], $constants['excluded_tax']);
		// stores the tax value, calculating it as the order class does
		$this->tax_manager->handle_value($row['products_tax'], $constants['tax']);
	}
	
	function _export($row) {
		static $previous_row = null;
		$export = array();
		
		// manage the shipping part at end of order process
		if ($row == null ||
		    ($row != null && $previous_row != null && $row['orders_id'] != $previous_row['orders_id'])) {
		 	$this->_manage_shipping($previous_row['orders_id']);
		 	// write the excluded tax amount ordered by account number (both sale and shipping operations)
		 	$this->_merge($this->_write_product_account($previous_row));
		 	// write all taxes values
		 	$this->_merge($this->_write_tax_account($previous_row));
		 	// write the customer part, included tax
		 	$this->_merge($this->_write_customer_account($previous_row));
		 	
		 	$this->tax_manager->clear();
		 	$this->sale_manager->clear();
		}
		
		if ($row != null) {
			$this->_handle_current_row($row);
			// store the old row
			$previous_row = $row;
		}
		
		return $this->export;
	}
	
	function export($query) {
		/**
		 * Export the formatted lines of data
		 * 
		 * @param query     a SQL Result of fetched informations
		 */
	
		header('Content-type: ' . $this->content_type);
    header('Content-disposition: attachment; filename=' . EXPORT_INVOICES_FILENAME);
    $this->export = array();
    
    while ($row = tep_db_fetch_array($query))
    	$this->_export($row);
    // now manage the last order with a null row value
    $this->_export(null);
    
    return implode("\r\n", $this->export);
	}
} 
?>
